/**
 * 	Premiere Learning Panel: JSX Module
 * 	for Premiere Pro v12.1+
 *
 *	@author: Florian Lange <f.lange@labor.digital>
 *	LABOR – Agentur für moderne Kommunikation GmbH, Mainz, Germany
 * 	https://labor.digital
 *
 */

//"that" is the global "this" of JSX for potential object inspection
var that = $.that;

var learnPanel = (function () {

	var globals = {
		selectedProjectItems: [],
		tutorialProjectNames: [],
		globalObjectForBinds: new Time() //to keep global bind events to our scope we don't bind to app but to any own object
	};

	var methods = {

		/**
		 * @description this will debug out in special Premiere App panel ( like typical notifications )
		 */
		dbgApp: function (message) {
			app.setSDKEventMessage(message, "info");
		},

		/**
		 * @param {trackItem} trackItem
		 * 
		 * @returns {array} array of components for debugging
		 */
		getAllTrackItemComponentsForDebug: function (trackItem) {
			var componentsOut = [];

			var components = methods.getTrackItemComponents(trackItem);
			var j;

			for (var i = 0; i < components.length; i++) {
				var component = components[i];

				var componentPrint = {};

				componentPrint.matchName = component.matchName;
				componentPrint.displayName = component.displayName;
				var properties = component.properties;
				componentPrint.properties = [];

				for (j = 0; j < properties.numItems; j++) {
					var componentParam = properties[j];
					var componentPartsPrint = {};
					componentPartsPrint.displayName = componentParam.displayName;
					componentPartsPrint.value = componentParam.getValue();
					componentPartsPrint.isTimeVarying = componentParam.isTimeVarying();
					//don't print out keyframes via componentParam.getKeys() here, e.g. when using this for a video track with lumetri settings it will throw error "bad argument list getKeys". why????

					componentPrint.properties.push(componentPartsPrint);
				}

				componentsOut.push(componentPrint);
			}

			return componentsOut;
		},

		/**
		 * @param {string} [path] full path
		 * 
		 * @returns {string} path
		 */
		openProject: function (path) {

			//with dialog
			if (!path) {

				var filterString = "";
				if (Folder.fs === 'Windows')
					filterString = "All files:*.*";

				var projToOpen = File.openDialog("Choose project:", filterString, false);

				if ((projToOpen) && projToOpen.exists) {
					path = projToOpen.fsName;

					app.openDocument(path,
						1, // suppress 'Convert Project' dialogs?
						0, // suppress 'Locate Files' dialogs?
						0, // suppress warning dialogs?
						1 // doNotAddToMRUList (available in PPro v12.1+)
					);

					projToOpen.close();
				}

				//without dialog
			} else {

				app.openDocument(path,
					1, // suppress 'Convert Project' dialogs?
					0, // suppress 'Locate Files' dialogs?
					0, // suppress warning dialogs?
					1 // doNotAddToMRUList (available in PPro v12.1+)
				);
			}

			return path;
		},

		/**
		 * @param {project} project
		 * @param {number} [saveFirst] 0 (default), 1
		 * @param {number} [promptUserIfDirty] 0 (default), 1
		 * 
		 * @returns {boolean} success
		 */
		closeProject: function (project, saveFirst, promptUserIfDirty) {
			//cast to number
			saveFirst = saveFirst ? 1 : 0;
			promptUserIfDirty = promptUserIfDirty ? 1 : 0;

			var success = project.closeDocument(saveFirst, promptUserIfDirty);

			return success;
		},

		/**
		 * @param {string} project
		 * 
		 * @returns {boolean}
		 */
		isCurrentProject: function (project) {
			var isCurrent = app.project.name == project;

			return isCurrent;
		},

		/**
		 * @returns {array} array of projects
		 */
		getOpenProjects: function () {
			var projectsArr = [];

			var projects = app.projects;

			//bug in v12.0: projects array was returning wrong list (only the current project multiple times). fixed in v12.1
			for (var i = 0; i < projects.numProjects; i++)
				projectsArr.push(projects[i]);

			return projectsArr;
		},

		/**
		 * @param {string} projectName
		 * 
		 * @returns {boolean|project} project or false
		 */
		getOpenProjectByName: function (projectName) {
			var project = false;

			//Get all projects
			var projects = methods.getOpenProjects();

			for (var i = 0; i < projects.length; i++) {
				if (projects[i].name == projectName)
					project = projects[i];
			}

			return project;
		},

		/**
		 * @param {string} [itemType] "sequence", "file"
		 * 
		 * @returns {array} array of projectItems
		 */
		getProjectItems: function (itemType) {
			projectItems = [];

			//possible ProjectItemType: CLIP, BIN, ROOT, FILE
			for (var i = 0; i < app.project.rootItem.children.numItems; i++) {
				/*
				 * TODO: bug in Premiere API: we always receive type=1 (CLIP), no matter what file.
				 * workaround: we check the metadata and look for "Column.Intrinsic.FilePath" -> this is only available in file items
				 *
				 *  new in PPro v12.1: there is a method for this ( isSequence() ) (not tested yet)
				 */
				if (!itemType || (itemType == "sequence" && app.project.rootItem.children[i].getProjectMetadata().indexOf("Column.Intrinsic.FilePath") === -1) || (itemType == "file"))
					projectItems.push(app.project.rootItem.children[i]);
			}

			return projectItems;
		},

		/**
		 * @returns {array} sequences. a clean array (instead of app.project.sequences which is mixed array and object)
		 */
		getSequences: function () {
			var sequences = [];

			var totalSequences = app.project.sequences.numSequences;

			for (var i = 0; i < totalSequences; i++)
				sequences.push(app.project.sequences[i]);

			return sequences;
		},

		/**
		 * @param {string} sequenceName
		 * 
		 * @returns {boolean|sequence} sequence or false
		 * 
		 * @description gets sequence by name in current active project
		 * 
		 * sequence name could exist multiple times. but each sequence has a unique id
		 */
		getSequenceByName: function (sequenceName) {
			var totalSequences = app.project.sequences.numSequences;
			var sequence = false;

			for (var i = 0; i < totalSequences; i++) {
				if (app.project.sequences[i].name === sequenceName) {
					sequence = app.project.sequences[i];

					//will return the first found sequence with the name
					break;
				}
			}

			return sequence;
		},

		/**
		 * @returns {boolean|sequence} sequence or false
		 */
		getActiveSequence: function () {
			var sequence = app.project.activeSequence;

			//when no active sequence exists it will be undefined
			if (typeof sequence == "undefined")
				sequence = false;

			return sequence;
		},

		/**
		 * @param {sequence} sequence
		 * 
		 * @returns {boolean}
		 */
		isActiveSequence: function (sequence) {
			var activeSequence = app.project.activeSequence;
			return activeSequence && activeSequence.sequenceID == sequence.sequenceID ? true : false;
		},

		/**
		 * @param {sequence} sequence
		 * 
		 * @returns {boolean} will return true if sequence could be opened
		 */
		openSequence: function (sequence) {
			//will return true even if the sequence was already open
			var result = sequence ? app.project.openSequence(sequence.sequenceID) : false;

			return result;
		},

		/**
		 * @param {sequence} sequence
		 * 
		 * @returns {boolean} will return true if sequence exists (no matter if already closed or not)
		 */
		closeSequence: function (sequence) {
			var result = false;

			//API sequence.close() returns undefined even when working
			if (sequence) {
				sequence.close();
				result = true;
			};

			return result;
		},

		/**
		 * @returns {boolean} true
		 */
		closeAllSequences: function () {
			var activeSequence;

			while (app.project.activeSequence) {
				activeSequence = app.project.activeSequence;
				if (activeSequence)
					activeSequence.close();
			}

			return true;
		},

		/**
		 * @param {sequence} [sequence] else will use active sequence
		 * @param {string} [trackType] "video", "audio"
		 * 
		 * @returns {array} array of tracks
		 */
		getTracks: function (sequence, trackType) {
			var tracks = [];
			var i;

			if (!sequence) {
				sequence = methods.getActiveSequence();
			}

			if (sequence) {

				if (!trackType || trackType == "video") {
					var videoTracks = sequence.videoTracks;
					for (i = 0; i < videoTracks.numTracks; i++)
						tracks.push(videoTracks[i]);
				}

				if (!trackType || trackType == "audio") {
					var audioTracks = sequence.audioTracks;
					for (i = 0; i < audioTracks.numTracks; i++)
						tracks.push(audioTracks[i]);
				}
			}

			return tracks;
		},

		/**
		 * @param {string} trackName
		 * 
		 * @returns {boolean|track} track or false
		 */
		getTrackByName: function (trackName) {
			var track = false;
			var tracks = methods.getTracks();

			for (var i = 0; i < tracks.length; i++) {
				if (tracks[i].name == trackName) {
					track = tracks[i];
					break;
				}
			}
			return track;
		},

		/**
		 * @param {string} [trackType] "video", "audio"
		 * @param {string} [layerName]
		 * 
		 * @description get trackItems in active sequence
		 * 
		 * @returns {array} array of trackItems (extended by: relatedTrack)
		 */
		getTrackItems: function (trackType, layerName) {
			var trackItems = [];

			var activeSequence = app.project.activeSequence;

			var track,
				trackItem,
				j;

			if (activeSequence) {
				var tracks = methods.getTracks(activeSequence, trackType);

				for (var i = 0; i < tracks.length; i++) {
					track = tracks[i];

					if (track && (!layerName || track.name == layerName)) {
						for (j = 0; j < track.clips.numItems; j++) {
							trackItem = track.clips[j];
							//extend: add releated track
							trackItem.relatedTrack = track;
							trackItems.push(trackItem);
						}
					}
				}
			}

			return trackItems;
		},

		/**
		 * @param {string} trackItemName
		 * @param {string} [trackType]
		 * @param {string} [layerName]
		 * 
		 * @returns {trackItem|boolean} trackItem or false
		 */
		getTrackItemByName: function (trackItemName, trackType, layerName) {
			var trackItem = false;
			var trackItems = methods.getTrackItems(trackType, layerName);

			for (var i = 0; i < trackItems.length; i++) {
				if (trackItems[i].name == trackItemName) {
					trackItem = trackItems[i];
					break;
				}
			}

			return trackItem;
		},

		/**
		 * @param {trackItem} trackItem
		 * @param {boolean} [selectLinkedItems] default false
		 * 
		 * @description select trackItem in active sequence
		 * 
		 * @returns {boolean} selected
		 */
		selectTrackItem: function (trackItem, selectLinkedItems) {
			var sequence = methods.getActiveSequence();
			var selected = false;

			if (sequence && trackItem) {
				selected = true;
				var selection = [trackItem];

				if (selectLinkedItems) {
					var linkedItems = trackItem.getLinkedItems();
					for (var i = 0; i < linkedItems.numItems; i++) {
						selection.push(linkedItems[i]);
					}
				}

				sequence.setSelection(selection);
			}

			return selected;
		},

		/**
		 * @returns {string|boolean} first found selected trackItem or false
		 */
		getFirstSelectedTrackItem: function () {
			var selectedTrackItems = methods.getSelectedTrackItems();
			return selectedTrackItems.length ? selectedTrackItems[0] : false;
		},

		/**
		 * @returns {array} array of selected trackItems
		 */
		getSelectedTrackItems: function () {
			var selectedTrackItems = [];
			var trackItem;

			var sequence = app.project.activeSequence;

			if (sequence) {
				var trackItems = methods.getTrackItems();

				for (var i = 0; i < trackItems.length; i++) {
					trackItem = trackItems[i];

					if (trackItem.isSelected())
						selectedTrackItems.push(trackItem);
				}
			}

			return selectedTrackItems;
		},

		/**
		 * @param {trackItem} trackItem
		 * 
		 * @returns {array} array of components
		 * 
		 * Info on components:
		 * component is an effect object, e.g. { id: 3, name: "Lumetri Farbe" }) and has methods to retrieve their properties
		 * it is totally different structured for trackItem vs trackItemQE!
		 */
		getTrackItemComponents: function (trackItem) {
			var components = [];
			var component;

			if (trackItem) {
				for (var i = 0; i < trackItem.components.numItems; i++) {
					component = trackItem.components[i];
					components.push(component);
				}
			}

			return components;
		},

		/**
		 * @param {component} component
		 * @param {number} paramIndex
		 * @param {string} [time] multiply the targeted frames by 8467200000 to get ticks and provide as string
		 */
		getComponentParamValue: function (component, paramIndex, time) {
			var value = typeof time !== "undefined" ? component.properties[paramIndex].getValueAtTime(time) : component.properties[paramIndex].getValue();

			return value;
		},

		/**
		 * @param {component} component
		 * @param {number} paramIndex
		 * 
		 * @returns {array} array of keyframes (a keyframe is an object {seconds: number, ticks: string})
		 */
		getComponentParamKeyframes: function (component, paramIndex) {
			var keyframes = [];

			if (component)
				keyframes = component.properties[paramIndex].getKeys();

			return typeof keyframes === "undefined" ? [] : keyframes;
		},

		/**
		 * @param {array} components array of components
		 * @param {number} matchName
		 * 
		 * @returns {component|boolean} component or false
		 */
		getComponentByMatchName: function (components, matchName) {
			var component = false;

			for (var i = 0; i < components.length; i++) {
				if (components[i].matchName == matchName) {
					component = components[i];
					break;
				}
			}

			return component;
		},

		/**
		 * @param {trackItem} trackItem
		 * @param {string} componentMatchName
		 * @param {number} paramIndex
		 * @param {string} [time] multiply the targeted frames by 8467200000 to get ticks and provide as string
		 * 
		 * @returns {any|boolean} value or false
		 */
		getTrackItemComponentValue: function (trackItem, componentMatchName, paramIndex, time) {
			var value = false;

			if (trackItem) {
				var components = methods.getTrackItemComponents(trackItem);
				var component = methods.getComponentByMatchName(components, componentMatchName);

				if (component)
					value = methods.getComponentParamValue(component, paramIndex, time);
			}

			return value;
		},

		/**
		 * @param {number|string} paramIndex param index number or name_of_constant
		 * 
		 * @returns {number} index. will return 0 if couldn't be resolved
		 */
		resolveComponentParamIndex: function (paramIndex) {
			if (typeof paramIndex === "string") {
				paramIndex = app.getConstant(paramIndex);

				//watch out: normalize since index is 1-based
				if (paramIndex !== -1)
					paramIndex = paramIndex - 1;
			}

			//robust behavior: if we can't resolve to a proper index we return 0 as a safe alternative
			if (typeof paramIndex === "undefined" || paramIndex === -1)
				paramIndex = 0;

			return paramIndex;
		},

		/**
		 * @param {array} paramIndices array of param index numbers or name_of_constants
		 * 
		 * @returns {array} array of indices as {number}
		 */
		resolveComponentParamIndices: function (paramIndices) {
			var paramIndicesResolved = [];

			for (var i = 0; i < paramIndices.length; i++)
				paramIndicesResolved.push(methods.resolveComponentParamIndex(paramIndices[i]));

			return paramIndicesResolved;
		},

		/**
		 * @returns {boolean|number} the current time of CTI in seconds or false
		 */
		getCti: function () {
			var cti = false;

			var activeSequence;
			if (app.project)
				activeSequence = app.project.activeSequence;

			if (activeSequence)
				cti = app.project.activeSequence.getPlayerPosition().seconds;

			return cti;
		},

		/**
		 * @returns {boolean|number} the current time of CTI in seconds or false
		 */
		getSourceCti: function () {
			var cti = false;

			if (app.sourceMonitor) {
				var val = app.sourceMonitor.getPosition().seconds;
				//bug in API: when no source is open it will return { seconds: -400000 }
				if (val != -400000)
					cti = val;
			}

			return cti;
		},

		/**
		 * @param {string} preferenceName
		 * @param {any} newValue
		 * @param {boolean} [allowCreateProperty]
		 * 
		 * @returns {object} 
		 * 		return.origVal = any
		 * 		return.changed = boolean
		 * 		return.notFound = boolean
		 * 		return.preferenceName = string
		 */
		changePreferenceByName: function (preferenceName, newValue, allowCreateProperty) {
			var origVal;
			var changed;
			var notFound = false;
			allowCreateProperty = allowCreateProperty || 0;

			var appProperties = app.properties;

			if (appProperties) {

				var propertyExists = app.properties.doesPropertyExist(preferenceName);
				var propertyIsReadOnly = app.properties.isPropertyReadOnly(preferenceName);
				var oldPropertyValue = app.properties.getProperty(preferenceName);

				origVal = oldPropertyValue;

				//convert string to real bool
				if (origVal === "false")
					origVal = false;
				else if (origVal === "true")
					origVal = true;

				//set new val
				// optional third parameter possible: 0 = non-persistent,  1 = persistent (default)
				if (propertyExists || allowCreateProperty) {

					if (!propertyIsReadOnly || allowCreateProperty) {

						appProperties.setProperty(preferenceName, newValue, 1, allowCreateProperty);

						var safetyCheck = app.properties.getProperty(preferenceName);
						changed = safetyCheck != oldPropertyValue;

					} else {

						changed = false;
					}

				} else {

					notFound = true;
				}

			} else {
				notFound = true;
			}

			return {
				origVal: origVal,
				changed: changed,
				notFound: notFound,
				preferenceName: preferenceName
			};
		},

		/**
		 * @param {?|number} markerIdentifier
		 * @param {string} sequenceName
		 * 
		 * @returns {marker|boolean} marker or false
		 */
		goToMarker: function (markerIdentifier, sequenceName) {
			var currentMarker = false;
			var projectSequence = methods.getSequenceByName(sequenceName);

			if (projectSequence) {

				var markers = projectSequence.markers;

				if (markers) {

					var numMarkers = markers.numMarkers;
					var index = (typeof markerIdentifier === 'number') ? markerIdentifier : numMarkers;

					if (numMarkers > 0) {

						var firstMarker = markers.getFirstMarker();
						var previousMarker = 1;

						if (firstMarker) {
							for (var i = 0; i < index; i++) {

								if (i === 0) {
									currentMarker = firstMarker;
								} else {
									currentMarker = markers.getNextMarker(previousMarker);
								}

								if (currentMarker) {
									projectSequence.setPlayerPosition(currentMarker.start.ticks);
									previousMarker = currentMarker;
								}
							}
						}
					}
				}
			}

			return currentMarker;
		},

		/**
		 * @param {string} windowName
		 * 
		 * @returns {boolean} isOpen
		 */
		isWindowOpen: function (windowName) {
			/*
			windowTypes = [
				"StoryPanel",
				"AudioClipMixer",
				"MiniAudoiMixer",
				"AudioMixer",
				"Captions",
				"Capture",
				"EditToTape",
				"Effect Controls",
				"Effects",
				"Graphics",
				"Essential Sounds",
				"Events",
				"History",
				"Info",
				"Services",
				"TitlerEditor",
				"Services",
				"Color", //= Lumetri Colors
				"Scopes",
				"MarkerList",
				"Media Browser",
				"MetadataEditor",
				"Program Monitor",
				"Progress",
				"Project",
				"ReferenceMonitor",
				"Source Monitor",
				"Timecode",
				"Tools",
				"Timeline",
				"TitlerProperties",
				"TitlerActions",
				"TitlerStyles",
				"TitlerTools",
				"Tools"
			];
			*/
			return app.isWindowVisible(windowName);
		}

	};

	// import the methods from common learnPanel module
	$.learnPanelMethods.extendMethods(methods);

	//extend the callFromJs
	methods.extendCallFromJs({

		/**
		 * @param {object} payload
		 * @param {string} payload.projectUrl
		 * @param {boolean} [payload.onlyOpenTutorialIfNoNontutorialProjectIsOpen] 
		 * 
		 * @returns {boolean} true
		 */
		openProjectAndCloseAndSaveOthers: function (payload) {
			var projectPath = payload.projectUrl;
			var onlyOpenTutorialIfNoNontutorialProjectIsOpen = payload.onlyOpenTutorialIfNoNontutorialProjectIsOpen || false;

			var projects = methods.getOpenProjects();
			var isNontutorialProjectOpen = false;

			if (onlyOpenTutorialIfNoNontutorialProjectIsOpen) {

				for (var i = 0; i < projects.length; i++) {
					var project = projects[i];

					if (globals.tutorialProjectNames.indexOf(project.name) === -1) {
						isNontutorialProjectOpen = true;
						break;
					}
				}
			}

			if (!onlyOpenTutorialIfNoNontutorialProjectIsOpen || !isNontutorialProjectOpen)
				//first: open the project (if it's already open will do nothing)
				methods.openProject(projectPath);

			//second: try to close others
			for (var i = 0; i < projects.length; i++) {

				//bug in v12.0: projects array was returning wrong list (only the current project multiple times). fixed in v12.1
				var project = projects[i];

				//if project is one of tutorials projects
				if (globals.tutorialProjectNames.indexOf(project.name) !== -1) {
					/**
					 * if project is not the one to be opened
					 * OR flag is set to only open if no other project is open and there is a non-tutorial open:
					 * 		close and ask for save
					 */
					if ((onlyOpenTutorialIfNoNontutorialProjectIsOpen && isNontutorialProjectOpen) || project.name != methods.getFilenameFromPath(projectPath))
						methods.closeProject(project, 0, 1);
				}
			}


			return true;
		},

		/**
		 * @param {object} payload
		 * @param {string} payload.tutorialProjectNames
		 * 
		 * @returns {boolean} true
		 */
		storeTutorialProjectNames: function (payload) {
			globals.tutorialProjectNames = payload.tutorialProjectNames;

			return true;
		},

		/**
		 * @returns {array} array of strings
		 */
		getOpenProjectsNames: function () {
			var projects = methods.getOpenProjects();
			var names = [];

			//bug in v12.0: projects array was returning wrong list (only the current project multiple times). fixed in v12.1
			for (var i = 0; i < projects.length; i++)
				names.push(projects[i].name);

			return names;
		},

		/**
		 * @returns {string|boolean} project name or false
		 */
		getCurrentProjectName: function () {
			var project = app.project;

			return project ? project.name : false;
		},

		/**
		 * @returns {boolean} true
		 */
		bindProjectItemSelected: function () {
			globals.globalObjectForBinds.bind('onSourceClipSelectedInProjectPanel', function (clips, viewID) {
				globals.selectedProjectItems = clips;
			});

			return true;
		},

		/**
		 * @returns {boolean} true
		 */
		unbindProjectItemSelected: function () {
			globals.globalObjectForBinds.unbind('onSourceClipSelectedInProjectPanel');

			//reset
			globals.selectedProjectItems = [];

			return true;
		},

		/**
		 * @returns {array} array of projectItems
		 */
		getSelectedProjectItems: function () {
			return globals.selectedProjectItems;
		},

		/**
		 * @param {object} payload
		 * @param {string} [payload.itemType]
		 * 
		 * @returns {boolean} true
		 */
		getProjectItemsCount: function (payload) {
			var itemType = payload.itemType;

			var projectItems = methods.getProjectItems(itemType);
			return projectItems.length;
		},

		/**
		 * @returns {number} the current time of CTI in timecode or false
		 */
		getSourceCti: function () {
			var cti = methods.getSourceCti();

			return cti;
		},

		/**
		 * @returns {array} array of strings
		 */
		getSequencesNames: function () {
			var sequences = methods.getSequences();
			var names = [];

			for (var i = 0; i < sequences.length; i++)
				names.push(sequences[i].name);

			return names;
		},

		/**
		 * @returns {boolean} true
		 */
		closeAllSequences: function () {
			return methods.closeAllSequences();
		},

		/**
		 * @param {object} payload
		 * @param {string} payload.sequenceName
		 * 
		 * @returns {boolean}
		 */
		openSequenceByName: function (payload) {
			var sequenceName = payload.sequenceName;

			return methods.openSequence(methods.getSequenceByName(sequenceName));
		},

		/**
		 * @param {object} payload
		 * @param {string} payload.sequenceName
		 * 
		 * @returns {boolean} true
		 */
		closeSequenceByName: function (payload) {
			var sequenceName = payload.sequenceName;

			return methods.closeSequence(methods.getSequenceByName(sequenceName));
		},

		/**
		 * @param {object} [payload]
		 * @param {string} [payload.clipName]
		 * @param {string} [payload.trackType]
		 * @param {string} [payload.layerName]
		 * 
		 * @description get trackItem count in active sequence
		 * 
		 * @returns {number} 
		 */
		getTrackItemsCount: function (payload) {
			var clipsCount;
			payload = payload || {};

			if (payload.clipName)
				clipsCount = methods.getTrackItemByName(payload.clipName, payload.trackType, payload.layerName) ? 1 : 0;
			else
				clipsCount = methods.getTrackItems(payload.trackType, payload.layerName).length;

			return clipsCount;
		},

		/**
		 * @param {object} [payload]
		 * @param {string} [payload.clipName] else it will get the selected track item
		 * 
		 * @description get trackItem properties in active sequence
		 * 
		 * @returns {extendedTrackItem|boolean} extendedTrackItem (trackItem + extra properties) or false
		 */
		getTrackItemProperties: function (payload) {
			var properties = false;

			var trackItemName = payload && payload.clipName ? payload.clipName : false;
			var trackItem = trackItemName ? methods.getTrackItemByName(trackItemName) : methods.getFirstSelectedTrackItem();

			properties = trackItem;
			if (trackItem) {
				//clone the object to be able to extend properties
				properties = JSON.parse(JSON.stringify(trackItem));

				//store the selected state
				properties.isSelected = trackItem.isSelected();

				//store speed properties
				properties.speed = trackItem.getSpeed();
				properties.isSpeedReversed = trackItem.isSpeedReversed();
			}

			return properties;
		},

		/**
		 * @param {object} payload
		 * @param {string} payload.clipName
		 * @param {boolean} [payload.selectLinkedItems]
		 * 
		 * @returns {boolean} selected
		 */
		selectTrackItemByName: function (payload) {
			var selected = methods.selectTrackItem(methods.getTrackItemByName(payload.clipName), payload.selectLinkedItems);

			return selected;
		},

		/**
		 * @param {object} payload
		 * @param {string} payload.clipName
		 * 
		 * @returns {boolean} selected
		 */
		isTrackItemSelected: function (payload) {
			var isSelected = false;
			var selectedItems = methods.getSelectedTrackItems();

			for (var i = 0; i < selectedItems.length; i++) {
				if (selectedItems[i].name == payload.clipName) {
					isSelected = true;
					break;
				}
			}

			return isSelected;
		},

		/**
		 * @param {object} payload
		 * @param {string} [payload.layerName]
		 * @param {string} payload.transitionName
		 * @param {string} payload.clipName
		 * 
		 * @returns {boolean} 
		 */
		hasTrackItemTransition: function (payload) {
			var hasTransition = false;
			var layerName = payload.layerName;
			var transitionName = payload.transitionName;
			var clipName = payload.clipName;

			var trackItem = methods.getTrackItemByName(clipName, false, layerName);
			var track = layerName ? methods.getTrackByName(layerName) : (trackItem ? trackItem.relatedTrack : false);

			if (track) {
				var transitions = track.transitions;
				for (var i = 0; i < transitions.numItems; i++) {
					var transition = transitions[i];

					//check for correct transitionName
					if (transition.matchName === transitionName) {

						if (trackItem) {
							//is transition within boundaries of trackItem
							if ((transition.start.seconds <= trackItem.start.seconds && transition.end.seconds > trackItem.start.seconds) ||
								(transition.start.seconds < trackItem.end.seconds && transition.end.seconds >= trackItem.end.seconds)) {
								hasTransition = true;
								break;
							}
						}
					}
				}
			}

			return hasTransition;
		},

		/**
		 * @returns {array} array of trackItems
		 */
		getTrackItemsProperties: function () {
			var properties = [];

			var trackItems = methods.getTrackItems("video");

			for (var i = 0; i < trackItems.length; i++)
				properties.push(trackItems[i]);

			return properties;
		},

		/**
		 * @param {object} payload
		 * @param {string} payload.componentMatchName
		 * @param {number|string} payload.paramIndex param index number or name_of_constant
		 * @param {string} [payload.trackItemName]
		 * @param {number} [payload.time]
		 * 
		 * @returns {any} 
		 */
		getTrackItemComponentValue: function (payload) {
			var componentMatchName = payload.componentMatchName;
			var paramIndex = methods.resolveComponentParamIndex(payload.paramIndex);
			var trackItemName = payload.clipName;

			var trackItem = trackItemName ? methods.getTrackItemByName(trackItemName) : methods.getFirstSelectedTrackItem();
			var value = methods.getTrackItemComponentValue(trackItem, componentMatchName, paramIndex, payload.time);

			return value;
		},

		/**
		 * @param {object} payload
		 * @param {string} payload.componentMatchName
		 * @param {array} payload.paramIndices array of param index numbers or name_of_constants
		 * @param {string} [payload.clipName]
		 * @param {number} [payload.time]
		 * 
		 * @returns {any} 
		 */
		getTrackItemComponentValues: function (payload) {
			var componentMatchName = payload.componentMatchName;
			var paramIndices = methods.resolveComponentParamIndices(payload.paramIndices);
			var trackItemName = payload.clipName;

			var trackItem = trackItemName ? methods.getTrackItemByName(trackItemName) : methods.getFirstSelectedTrackItem();

			var values = [];
			for (var i = 0; i < paramIndices.length; i++) {
				var value = methods.getTrackItemComponentValue(trackItem, componentMatchName, paramIndices[i], payload.time);
				values.push(value);
			}

			return values;
		},

		/**
		 * @param {object} payload
		 * @param {string} payload.componentMatchName
		 * @param {number|string} payload.paramIndex param index number or name_of_constant
		 * @param {array} payload.times array of times in seconds to get the value at
		 * @param {string} [payload.trackItemName]
		 * 
		 * @returns {array} array of values (same length as times)
		 */
		getTrackItemComponentValuesByTime: function (payload) {
			var componentMatchName = payload.componentMatchName;
			var paramIndex = methods.resolveComponentParamIndex(payload.paramIndex);
			var trackItemName = payload.clipName;
			var times = payload.times;

			var values = [];

			var trackItem = trackItemName ? methods.getTrackItemByName(trackItemName) : methods.getFirstSelectedTrackItem();
			for (var i = 0; i < times.length; i++) {
				values.push(methods.getTrackItemComponentValue(trackItem, componentMatchName, paramIndex, times[i]));
			}

			return values;
		},

		/**
		 * @param {object} payload
		 * @param {string} payload.componentMatchName
		 * @param {number|string} payload.paramIndex param index number or name_of_constant
		 * @param {string} [payload.trackItemName]
		 * 
		 * @returns {array} array of keyframes
		 */
		getTrackItemKeyframes: function (payload) {
			var componentMatchName = payload.componentMatchName;
			var paramIndex = methods.resolveComponentParamIndex(payload.paramIndex);
			var trackItemName = payload.clipName;

			var trackItem = trackItemName ? methods.getTrackItemByName(trackItemName) : methods.getFirstSelectedTrackItem();
			var keyframes = methods.getComponentParamKeyframes(methods.getComponentByMatchName(methods.getTrackItemComponents(trackItem), componentMatchName), paramIndex);

			return keyframes;
		},

		/**
		 * @returns {number} the current time of CTI in frames or false
		 */
		getCti: function () {
			var cti = methods.getCti();

			return cti;
		},

		/**
		 * @param {object} [payload]
		 * @param {string} [payload.clipName]
		 * @param {string} [payload.layerName]
		 * 
		 * @returns {boolean} 
		 */
		isCtiOverTrackItem: function (payload) {
			var isOver = false;
			var cti = methods.getCti();

			var layerName = payload.layerName;
			var clipName = payload.clipName;
			var trackItem;

			if (cti !== false) {


				if (layerName && !clipName) {
					//if only layerName is provided: check for all clips on that layer

					var trackItems = methods.getTrackItems(false, layerName);
					for (var i = 0; i < trackItems.length; i++) {
						trackItem = trackItems[i];

						if (cti >= trackItem.start.seconds && cti <= trackItem.end.seconds)
							isOver = true;
					}


				} else {
					//clipName is provided

					trackItem = methods.getTrackItemByName(payload.clipName, false, layerName);

					if (trackItem && cti >= trackItem.start.seconds && cti <= trackItem.end.seconds)
						isOver = true;
				}
			}

			return isOver;
		},

		/**
		 * @param {object} payload
		 * @param {string} payload.sequenceName
		 * @param {number} payload.index
		 * 
		 * @returns {boolean}
		 */
		goToMarkerByIndex: function (payload) {
			var sequenceName = payload.sequenceName;
			var index = payload.index;

			return methods.goToMarker(index, sequenceName) ? true : false;

		},

		/**
		 * @param {object} payload
		 * @param {string} payload.sequenceName
		 * @param {string} payload.markerName
		 * 
		 * @returns {boolean}
		 */
		goToMarkerByName: function (payload) {
			var sequenceName = payload.sequenceName;
			var markerName = payload.markerName;

			return methods.goToMarker(markerName, sequenceName) ? true : false;
		},

		/**
		 * @returns {boolean} true
		 */
		showOnboarding: function () {
			app.showOnboarding(3);

			return true;
		},

		/**
		 * @param {object} payload
		 * @param {string} payload.panelName
		 * 
		 * @returns {boolean} 
		 */
		isWindowOpen: function (payload) {
			return methods.isWindowOpen(payload.panelName);
		},

		/**
		 * @param {string} newPath
		 * 
		 * @returns {object} preferenceResult from changePreferenceByName()
		 */
		setPrefImportPath: function (newPath) {
			newPath = methods.getCleanPath(newPath);

			//that's the path which will open the directory when e.g. the user doubleclicks on the project panel to import custom media
			var result = methods.changePreferenceByName("be.prefs.last.used.directory", newPath, 1);
			return result;
		},

		/**
		 * @param {string} newVal
		 * 
		 * @returns {object} preferenceResult from changePreferenceByName()
		 */
		setPrefImportWorkspace: function (newVal) {
			//that's the setting in the menu "Window > Workspaces > Import Workspace from project"
			var result = methods.changePreferenceByName("FE.Prefs.ImportWorkspace", newVal);
			return result;
		},

		/**
		 * @param {string} newVal
		 * 
		 * @returns {object} preferenceResult from changePreferenceByName()
		 */
		setPrefRippleSequenceMarkers: function (newVal) {
			//that's the setting in the menu "Markers > Ripple Sequence Markers"
			var result = methods.changePreferenceByName("FE.Prefs.Timeline.SequenceMarker.Ripple.Mode", newVal);
			return result;
		},

		/**
		 * @param {string} extensionId
		 * 
		 * @returns {boolean} true
		 */
		setExtensionPersistent: function (extensionId) {
			app.setExtensionPersistent(extensionId, 1);

			return true;
		},

		getResourcePaths: function () {
			return {
				//path must have trailing slash
				learnPanelContentDirPath: methods.getPathWithSlashes(app.learnPanelContentDirPath),

				//path must have trailing slash
				learnPanelExampleProjectDirPath: methods.getPathWithSlashes(app.learnPanelExampleProjectDirPath)
			}
		},

		test: function (payload) {
			//see the AEFT JSX for info on howto debug
			return true;
		}

	});

	//shortcuts
	var dbgApp = methods.dbgApp;
	var dbg = methods.dbg;

	//return public methods & properties
	return {
		callFromJs: methods.callFromJs
	};

})();

//make accessible for JS via $
$.callFromJs = learnPanel.callFromJs;